/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	$Id: pgpRPCKeyBack.c,v 1.55.2.2 2001/08/16 17:20:31 pbj Exp $
____________________________________________________________________________*/

#include "pgpPubTypes.h"
#include "pgpErrors.h"
#include "pgpFileSpec.h"
#include "pgpUtilities.h"
#include "pgpKeys.h"
#include "pgpPublicKey.h"
#include "pgpRPCMsg.h"
#include "pgpKeyPriv.h"

#if PGP_WIN32
#include <windows.h>
#endif

static PGPInt32 connect_ref = 0;

#if PGP_MACINTOSH
#pragma warn_unusedarg off
#endif

	PGPError
pgpConnect_backRPC()
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_Connect_MSG, 0);
	pack_int32(&pkt, PGPMSG_PROTOCOL_REV);
#if PGP_WIN32
	pack_uint32(&pkt, GetCurrentProcessId());
	pack_uint32(&pkt, pgpSDKGetFrontEndEventHandle());
#else
	pack_uint32(&pkt, 0);
	pack_uint32(&pkt, 0);
#endif

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
/*	pgpAssert( IsntPGPError(status) ); */
	if (IsntPGPError(status)) {
		connect_ref = unpack_int32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpReconnect_backRPC()
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_Reconnect_MSG, connect_ref);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpDisconnect_backRPC()
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	if (connect_ref == 0)
		return kPGPError_NoErr;

	pack_init(&pkt, kPGP_Disconnect_MSG, connect_ref);
	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	connect_ref = 0;
	return status;
}

	PGPError
pgpFetchObjectData_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPByte **bufPtr,		/* return parameters */
	PGPSize *bufSize)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_FetchObjectData_MSG, connect_ref);
	pack_int32(&pkt, id);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)bufPtr, bufSize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpGetKeyByKeyID_backRPC(
	PGPContextRef ctx,
	PGPUInt32 dbid,
	PGPKeyID const *keyIDIn,
	PGPBoolean	dummyOK,
	PGPBoolean	deletedOK,
	PGPUInt32	*outID)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_GetKeyByKeyID_MSG, connect_ref);
	pack_uint32(&pkt, dbid);
	pack_opaque(&pkt, (PGPByte *)keyIDIn, sizeof(*keyIDIn));
	pack_bool(&pkt, dummyOK);
	pack_bool(&pkt, deletedOK);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*outID = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpKeyEncrypt_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPByte const *inbuf,
	PGPSize inbuflen,
	PGPPublicKeyMessageFormat format,
	PGPByte **outbuf,
	PGPSize *outbuflen)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_KeyEncrypt_MSG, connect_ref);
	pack_uint32(&pkt, id);
	pack_opaque(&pkt, (PGPByte *)inbuf, inbuflen);
	pack_uint32(&pkt, (PGPUInt32)format);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)outbuf, outbuflen);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpKeyDecrypt_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	const PGPByte *passphrase,
	PGPSize pplen,
	PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	const PGPByte *inbuf,
	PGPSize inbuflen,
	PGPPublicKeyMessageFormat format,
	PGPByte **outbuf,
	PGPSize *outbuflen)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init_secure(&pkt, kPGP_KeyDecrypt_MSG, connect_ref);
	pack_int32(&pkt, id);
	pack_opaque(&pkt, passphrase, pplen);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);
	pack_opaque(&pkt, (PGPByte *)inbuf, inbuflen);
	pack_uint32(&pkt, (PGPUInt32)format);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)outbuf, outbuflen);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPInt32
pgpKeyVerify_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPByte const *inbuf,
	PGPSize inbuflen,
	PGPHashAlgorithm hashalg,
	PGPByte const *hash,
	PGPSize hashlen,
	PGPPublicKeyMessageFormat format)
{
	PGPInt32 status, n;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_KeyVerify_MSG, connect_ref);
	pack_uint32(&pkt, id);
	pack_opaque(&pkt, (PGPByte *)inbuf, inbuflen);
	pack_uint32(&pkt, (PGPUInt32)hashalg);
	pack_opaque(&pkt, (PGPByte *)hash, hashlen);
	pack_uint32(&pkt, (PGPUInt32)format);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		n = unpack_int32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return n;
}

	PGPError
pgpKeySign_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPByte const *passphrase,
	PGPSize pplen,
	PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut,
	PGPBoolean cacheGlobal,
	PGPHashAlgorithm hashalg,
	PGPByte const *hash,
	PGPSize hashlen,
	PGPPublicKeyMessageFormat format,
	PGPByte **outbuf,
	PGPSize *outbuflen)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init_secure(&pkt, kPGP_KeySign_MSG, connect_ref);
	pack_int32(&pkt, id);
	pack_opaque(&pkt, (PGPByte *)passphrase, pplen);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);
	pack_uint32(&pkt, (PGPUInt32)hashalg);
	pack_opaque(&pkt, (PGPByte *)hash, hashlen);
	pack_uint32(&pkt, (PGPUInt32)format);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)outbuf, outbuflen);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPBoolean
pgpSecPassphraseOK_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPByte const *passphrase,
	PGPSize pplen,
	PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut,
	PGPBoolean cacheGlobal )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;
	PGPBoolean b;

	(void)ctx;
	pack_init_secure(&pkt, kPGP_SecPassphraseOK_MSG, connect_ref);
	pack_uint32(&pkt, id);
	pack_opaque(&pkt, (PGPByte *)passphrase, pplen);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsPGPError(status))
		b = FALSE;
	else {
		b = unpack_bool(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return b;
}

	PGPError
pgpPurgePassphraseCache_backRPC(
	PGPContextRef ctx)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_PurgePassphraseCache_MSG, connect_ref);
	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status))
		PGPFreeData(rplypkt.base);
	return status;
}


	PGPError
pgpKeyMaxSizes_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPUInt32 *maxEncryption,
	PGPUInt32 *maxDecryption,
	PGPUInt32 *maxSignature,
	PGPPublicKeyMessageFormat format)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;
	
	(void)ctx;
	pack_init(&pkt, kPGP_KeyMaxSizes_MSG, connect_ref);
	pack_uint32(&pkt, id);
	pack_bool(&pkt, (PGPBoolean)(IsntNull(maxEncryption)));
	pack_bool(&pkt, (PGPBoolean)(IsntNull(maxDecryption)));
	pack_bool(&pkt, (PGPBoolean)(IsntNull(maxSignature)));
	pack_uint32(&pkt, (PGPUInt32)format);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPUInt32 mymax;
		mymax = unpack_uint32(&rplypkt);
		if( IsntNull( maxEncryption ) )
			*maxEncryption = mymax;
		mymax = unpack_uint32(&rplypkt);
		if( IsntNull( maxDecryption ) )
			*maxDecryption = mymax;
		mymax = unpack_uint32(&rplypkt);
		if( IsntNull( maxSignature ) )
			*maxSignature = mymax;
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpSecProperties_backRPC( PGPContextRef context, PGPUInt32 id,
	PGPBoolean *needsPassphrase, PGPBoolean *isSecretShared,
	PGPCipherAlgorithm *lockAlg, PGPUInt32 *lockBits )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_SecProperties_MSG, connect_ref);
	pack_uint32(&pkt, id);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*needsPassphrase = unpack_uint32(&rplypkt);
		*isSecretShared = unpack_uint32(&rplypkt);
		*lockAlg = (PGPCipherAlgorithm) unpack_uint32(&rplypkt);
		*lockBits = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpFetchKeyInfo_backRPC(
	PGPContextRef ctx,
	PGPUInt32 id,
	PGPByte **bufPtr,		/* return parameters */
	PGPSize *bufSize)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_FetchKeyInfo_MSG, connect_ref);
	pack_int32(&pkt, id);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)bufPtr, bufSize);
		PGPFreeData(rplypkt.base);
	}

	return status;
}

	PGPError
pgpOpenKeyDBFile_backRPC(
	PGPContextRef ctx,
	PGPOpenKeyDBFileOptions openFlags,
	PFLFileSpecRef pub, 
	PFLFileSpecRef priv,
	PGPUInt32 *kdbid,		/* return parameters */
	PGPUInt32 *numKeys,
	PGPUInt32 **keyBuffer,
	PGPSize *bufSize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctx;
	pack_init(&pkt, kPGP_OpenKeyDBFile_MSG, connect_ref);
	pack_int32(&pkt, openFlags);
	pack_filespec(&pkt, pub);
	pack_filespec(&pkt, priv);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*kdbid = unpack_int32(&rplypkt);
		*numKeys = unpack_int32(&rplypkt);
		unpack_opaque_alloc(&rplypkt, (void **)keyBuffer, bufSize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpImportKeyBinary_backRPC(
	PGPContextRef			context,
	PGPByte *				buffer,
	PGPSize					length,
	PGPUInt32 *				kdbid,
	PGPUInt32 *				numKeys,
	PGPUInt32 **			keyArray,
	PGPSize *				keyArraySize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_ImportKeyBinary_MSG, connect_ref);
	pack_opaque(&pkt, buffer, length);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*kdbid = unpack_uint32(&rplypkt);
		*numKeys = unpack_uint32(&rplypkt);
		unpack_opaque_alloc(&rplypkt, (void **)keyArray, keyArraySize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpKeyDBFlush_backRPC( PGPContextRef context, PGPUInt32 id,
	PGPUInt32 **changedkeylist, PGPSize *changedkeylistsize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_KeyDBFlush_MSG, connect_ref);
	pack_int32(&pkt, id);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)changedkeylist, changedkeylistsize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	void
pgpFreeKeyDB_backRPC( PGPContextRef context, PGPUInt32 id )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_FreeKeyDB_MSG, connect_ref);
	pack_int32(&pkt, id);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status))
		PGPFreeData(rplypkt.base);
	return;
}

	PGPError
pgpSetKeyEnabled_backRPC( PGPContextRef context, PGPUInt32 id, PGPBoolean enable  )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_SetKeyEnabled_MSG, connect_ref);
	pack_int32(&pkt, id);
	pack_bool(&pkt, enable);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status))
		PGPFreeData(rplypkt.base);
	return status;
}

	PGPError
pgpSetKeyAxiomatic_backRPC(
	PGPContextRef	context,
	PGPUInt32		id,
	PGPBoolean		setAxiomatic,
	PGPBoolean		checkPassphrase,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_SetKeyAxiomatic_MSG, connect_ref);
	pack_int32(&pkt, id);
	pack_bool(&pkt, setAxiomatic);
	pack_bool(&pkt, checkPassphrase);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status))
		PGPFreeData(rplypkt.base);
	return status;
}

	PGPError
pgpPropagateTrust_backRPC( PGPContextRef context, PGPUInt32 setid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPUInt32 altid,
	PGPUInt32 const timenow, PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_PropagateTrust_MSG, connect_ref);
	pack_uint32(&pkt, setid);
	pack_opaque(&pkt, keylist, keylistsize);
	pack_uint32(&pkt, altid);
	pack_uint32(&pkt, timenow);

	if (keylist) PGPFreeData(keylist);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)changedkeylist, changedkeylistsize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpCheckKeyRingSigs_backRPC( PGPContextRef context, PGPUInt32 setid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPUInt32 altid,
	PGPBoolean checkAll, PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_CheckKeyRingSigs_MSG, connect_ref);
	pack_uint32(&pkt, setid);
	pack_opaque(&pkt, keylist, keylistsize);
	pack_uint32(&pkt, altid);
	pack_bool(&pkt, checkAll);

	if (keylist) PGPFreeData(keylist);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)changedkeylist, changedkeylistsize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpPrepareToCheckKeyRingSigs_backRPC( PGPContextRef context, PGPUInt32 setid,
	PGPUInt32 *keylist, PGPSize keylistsize, PGPUInt32 altid,
	PGPBoolean checkAll, PGPUInt32 *nsigs, PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_PrepareToCheckKeyRingSigs_MSG, connect_ref);
	pack_uint32(&pkt, setid);
	pack_opaque(&pkt, keylist, keylistsize);
	pack_uint32(&pkt, altid);
	pack_bool(&pkt, checkAll);

	if (keylist) PGPFreeData(keylist);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*nsigs = unpack_uint32(&rplypkt);
		unpack_opaque_alloc(&rplypkt, (void **)changedkeylist, changedkeylistsize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpCheckSig_backRPC( PGPContextRef context, PGPUInt32 setid, PGPUInt32 objid,
	PGPUInt32 altid, PGPBoolean checkAll, PGPBoolean revocationonly,
	PGPBoolean *handled, PGPBoolean *changed, PGPBoolean *verified )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_CheckSig_MSG, connect_ref);
	pack_uint32(&pkt, setid);
	pack_uint32(&pkt, objid);
	pack_uint32(&pkt, altid);
	pack_bool(&pkt, checkAll);
	pack_bool(&pkt, revocationonly);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*handled = unpack_bool(&rplypkt);
		*changed = unpack_bool(&rplypkt);
		*verified = unpack_bool(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpDoGenerateKey_backRPC (
	PGPContextRef	context,
	PGPUInt32		keydbID,
	PGPUInt32		masterkeyID,
	PGPByte			pkalg,
	PGPUInt32		bits,
	PGPTime			creationDate,
	PGPUInt16		expirationDays,
	char const *	name,
	PGPSize			name_len, 
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	char const *	masterpass, 
	PGPSize			masterpassLength,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPEventHandlerProcPtr progress,
	PGPUserValue	userValue,
	PGPBoolean		fastgen,
	PGPBoolean		checkentropy,
	PGPBoolean		usetoken,
	PGPUInt32		tokenid,
	PGPUInt32 *		adkkeylist,
	PGPSize			adkkeylistsize,
	PGPByte			adkclass,
	PGPUInt32 *		rakkeylist,
	PGPSize			rakkeylistsize,
	PGPByte			rakclass,
	PGPCipherAlgorithm const * prefalg,
	PGPSize			prefalgLength,
	PGPByte const * prefkeyserv,
	PGPSize prefkeyservLength,
	PGPUInt32 keyflags,
	PGPBoolean fkeyflags,
	PGPUInt32 keyservprefs,
	PGPBoolean fkeyservprefs,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength,
	PGPUInt32 *		newKey)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_DoGenerateKey_MSG, connect_ref);
	pack_uint32(&pkt, keydbID);
	pack_uint32(&pkt, masterkeyID);
	pack_int32(&pkt, pkalg);
	pack_uint32(&pkt, bits);
	pack_uint32(&pkt, creationDate);
	pack_uint32(&pkt, expirationDays);
	pack_opaque(&pkt, (void *)name, name_len);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, passphraseIsKey);
	pack_opaque(&pkt, (void *)masterpass, masterpassLength);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);
	pack_uint32(&pkt, (PGPUInt32)progress);
	pack_uint32(&pkt, (PGPUInt32)userValue);
	pack_bool(&pkt, fastgen);
	pack_bool(&pkt, checkentropy);
	pack_bool(&pkt, usetoken);
	pack_uint32(&pkt, tokenid);
	pack_opaque(&pkt, adkkeylist, adkkeylistsize);
	pack_byte(&pkt, adkclass);
	pack_opaque(&pkt, rakkeylist, rakkeylistsize);
	pack_byte(&pkt, rakclass);
	pack_opaque(&pkt, (void *)prefalg, prefalgLength);
	pack_opaque(&pkt, prefkeyserv, prefkeyservLength);
	pack_uint32(&pkt, keyflags);
	pack_bool(&pkt, fkeyflags);
	pack_uint32(&pkt, keyservprefs);
	pack_bool(&pkt, fkeyservprefs);

	if (adkkeylist) PGPFreeData(adkkeylist);
	if (rakkeylist) PGPFreeData(rakkeylist);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void *)newobjs, newobjslength);
		*newKey = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpCreateKeypair_backRPC (
	PGPContextRef	context,
	PGPUInt32		keydbID,
	PGPByte *		seckeyBuf,
	PGPSize			seckeySize,
	PGPByte *		keyspecBuf,
	PGPSize			keyspecSize,
	char const *	name,
	PGPSize			name_len, 
	PGPUInt32 *		adklist,
	PGPSize			adklistsize,
	PGPByte			adkclass,
	PGPUInt32 *		raklist,
	PGPSize			raklistsize,
	PGPByte			rakclass,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength,
	PGPUInt32 *		newkeyid )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_CreateKeypair_MSG, connect_ref);
	pack_uint32(&pkt, keydbID);
	pack_opaque(&pkt, (void *)seckeyBuf, seckeySize);
	pack_opaque(&pkt, (void *)keyspecBuf, keyspecSize);
	pack_opaque(&pkt, (void *)name, name_len);
	pack_opaque(&pkt, adklist, adklistsize);
	pack_byte(&pkt, adkclass);
	pack_opaque(&pkt, raklist, raklistsize);
	pack_byte(&pkt, rakclass);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, passphraseIsKey);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	if (adklist) PGPFreeData(adklist);
	if (raklist) PGPFreeData(raklist);
	if (keyspecBuf) PGPFreeData(keyspecBuf);
	if (seckeyBuf) PGPFreeData(seckeyBuf);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void *)newobjs, newobjslength);
		*newkeyid = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpCreateSubkeypair_backRPC (
	PGPContextRef	context,
	PGPUInt32		masterkeyID,
	PGPByte *		seckeyBuf,
	PGPSize			seckeySize,
	PGPByte *		keyspecBuf,
	PGPSize			keyspecSize,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		passphraseIsKey,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength,
	PGPUInt32 *		newkeyid )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_CreateSubkeypair_MSG, connect_ref);
	pack_uint32(&pkt, masterkeyID);
	pack_opaque(&pkt, (void *)seckeyBuf, seckeySize);
	pack_opaque(&pkt, (void *)keyspecBuf, keyspecSize);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, passphraseIsKey);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	if (keyspecBuf) PGPFreeData(keyspecBuf);
	if (seckeyBuf) PGPFreeData(seckeyBuf);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void *)newobjs, newobjslength);
		*newkeyid = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


PGPBoolean enable_checking = FALSE;

	PGPError
pgpCountTokens_backRPC (
	PGPContextRef	context,
	PGPUInt32 *		numTokens
	)

{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;


	pack_init_secure(&pkt, kPGP_CountTokens_MSG, connect_ref);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*numTokens = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}


	return status;
}


   	PGPError
pgpGetTokenInfo_backRPC (
	PGPContextRef	context,
    PGPUInt32  tokNumber, PGPTokenInfo *info )

{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;
    PGPUInt32 client_size = info->size;   /* sizeof(PGPTokenInfo) for client */

	(void)context;
    memset( info, 0, sizeof(*info) );

	pack_init_secure(&pkt, kPGP_GetTokenInfo_MSG, connect_ref);
	pack_uint32(&pkt, tokNumber);
    pack_uint32(&pkt, client_size);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
        PGPByte *p;
        PGPSize p_size;

        /* Get data from SDK.
           Begin of base structure PGPTokenInfo */

        /* sizeof(PGPTokenInfo) for SDK */
        info->size = unpack_uint32(&rplypkt);

       	info->bRsa = unpack_bool(&rplypkt);
        pgpAssert( info->bRsa == 1 || info->bRsa == 0 );

        info->minRsaKeySize = unpack_uint32(&rplypkt);
        info->maxRsaKeySize = unpack_uint32(&rplypkt);
        pgpAssert( info->maxRsaKeySize >= info->minRsaKeySize );
        info->minPinLen = unpack_uint32(&rplypkt);
        info->maxPinLen = unpack_uint32(&rplypkt);
        pgpAssert( info->maxPinLen >= info->minPinLen );

        p_size = 0;
		unpack_opaque_alloc(&rplypkt, (void **)&p, &p_size);
        if( p_size )
            strncpy( info->manufacturerID, p, p_size );
        PGPFreeData( p );

        p_size = 0;
		unpack_opaque_alloc(&rplypkt, (void **)&p, &p_size);
        if( p_size )
            strncpy( info->model, p, p_size );
        PGPFreeData( p );
        
        p_size = 0;
		unpack_opaque_alloc(&rplypkt, (void **)&p, &p_size);
        if( p_size )
            strncpy( info->serialNumber, p, p_size );
        PGPFreeData( p );

        info->numPubKeys = unpack_uint32(&rplypkt);
        info->numPrivKeys = unpack_uint32(&rplypkt);

        /* End of base structure PGPTokenInfo, therefore: */
        pgpAssert( info->size==0 || info->size/*SDK size*/ == client_size );

		PGPFreeData(rplypkt.base);
	}

    return status;
}

	PGPError
pgpDeleteKeyOnToken_backRPC( PGPContextRef	context, 
							PGPUInt32 keydbID, 
							const PGPKeyID *keyID, 
							PGPUInt32 tokenNumber, 
							PGPByte const *pin, PGPSize pinLen )  
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_DeleteKeyOnToken_MSG, connect_ref);
	
	pack_uint32(&pkt, keydbID);
	pack_opaque(&pkt, keyID, sizeof(PGPKeyID));
	pack_uint32(&pkt, tokenNumber);
	pack_opaque(&pkt, (void *)pin, pinLen);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) 
		PGPFreeData(rplypkt.base);
	return status;
}

	PGPError
pgpWipeToken_backRPC (
	PGPContextRef	context,
	PGPUInt32 		tokNumber,
	PGPByte const *	passphrase,
	PGPSize			passphraseLength
	)

{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_WipeToken_MSG, connect_ref);
	pack_uint32(&pkt, tokNumber);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpTokenPassphraseIsValid_backRPC (
	PGPContextRef	context,
	PGPUInt32 		tokNumber,
	PGPByte const *	passphrase,
	PGPSize			passphraseLength
	)

{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_TokenPassphraseIsValid_MSG, connect_ref);
	pack_uint32(&pkt, tokNumber);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpCopyKeyToToken_backRPC (
	PGPContextRef	context,
	PGPUInt32		objid,
	PGPUInt32		tokNumber,
	PGPBoolean		isMaster, 
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	char const *	PIN,
	PGPSize			PINlength,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_CopyKeyToToken_MSG, connect_ref);
	pack_uint32(&pkt, objid);
	pack_uint32(&pkt, tokNumber);
	pack_bool(&pkt, isMaster);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_opaque(&pkt, (void *)PIN, PINlength);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

   	PGPError
pgpTokenImportX509_backRPC (
	PGPContextRef	context,
	const PGPByte         *keyID,
    const PGPByte         *userID,    PGPSize     userID_len, 
    const PGPByte         *x509,      PGPSize     x509_len,
    const PGPByte         *password,  PGPSize     passwordLength )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init_secure(&pkt, kPGP_TokenImportX509_MSG, connect_ref);
	pack_uint32(&pkt, *(PGPUInt32*)keyID);
    pack_uint32(&pkt, ((PGPUInt32*)keyID)[1]);
    pack_opaque(&pkt, userID, userID_len);
	pack_opaque(&pkt, password, passwordLength);
	pack_opaque(&pkt, (void *)x509, x509_len);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

    
PGPError                
pgpTokenPutKeyContainer_backRPC( 
     PGPContextRef context, const PGPByte *keyID, 
     const PGPByte *password, PGPSize passwordSize, 
     const PGPByte *cont, PGPSize contSize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init_secure(&pkt, kPGP_TokenPutKeyContainer_MSG, connect_ref);
	pack_uint32(&pkt, *(PGPUInt32*)keyID);
    pack_uint32(&pkt, ((PGPUInt32*)keyID)[1]);
    pack_opaque(&pkt, cont, contSize);
    pack_opaque(&pkt, password, passwordSize);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status) ) {
		PGPFreeData(rplypkt.base);
	}

	return status;
}

PGPError                
pgpTokenGetKeyContainer_backRPC( 
     PGPContextRef context, const PGPByte *keyID, 
     const PGPByte *password, PGPSize passwordSize,
     PGPByte **contOut, PGPSize *contOutSize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init_secure(&pkt, kPGP_TokenGetKeyContainer_MSG, connect_ref);
	pack_uint32(&pkt, *(PGPUInt32*)keyID);
    pack_uint32(&pkt, ((PGPUInt32*)keyID)[1]);
    pack_opaque(&pkt, password, passwordSize);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status) && contOut ) {
		unpack_opaque_alloc( &rplypkt, (void **)contOut, contOutSize );
		PGPFreeData(rplypkt.base);
	}

	return status;
}


    PGPError 
pgpSetPKCS11DrvFile_backRPC( PGPByte *module )  
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init_secure(&pkt, kPGP_SetPKCS11DrvFile_MSG, connect_ref);
	pack_opaque(&pkt, (void *)module, strlen(module)+1);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpAddUserID_backRPC (
	PGPContextRef	context,
	PGPUInt32		objid,
	PGPBoolean		isAttribute,
	PGPAttributeType attributeType,
	char const *	userIDData,
	PGPSize		   	userIDLength,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32 cacheTimeOut,
	PGPBoolean cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength)

{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_AddUserID_MSG, connect_ref);
	pack_uint32(&pkt, objid);
	pack_bool(&pkt, isAttribute);
	pack_int32(&pkt, attributeType);
	pack_opaque(&pkt, (void *)userIDData, userIDLength);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpCertifyUserID_backRPC(
	PGPContextRef	context,
	PGPUInt32		useridid,
	PGPUInt32		certifying_keyid,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPBoolean		exportable,
	PGPTime			creationDate,
	PGPUInt32		expiration,
	PGPByte			trustDepth,
	PGPByte			trustValue,
	char const *	sRegExp,
	PGPSize			sRegExpLength,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength
	)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_CertifyUserID_MSG, connect_ref);
	pack_uint32(&pkt, useridid);
	pack_uint32(&pkt, certifying_keyid);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);
	pack_bool(&pkt, exportable);
	pack_uint32(&pkt, creationDate);
	pack_uint32(&pkt, expiration);
	pack_byte(&pkt, trustDepth);
	pack_byte(&pkt, trustValue);
	pack_opaque(&pkt, (void *)sRegExp, sRegExpLength);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpRevokeSig_backRPC (
	PGPContextRef	context,
	PGPUInt32		sigid,
	char const *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPUInt32 **	newobjs,
	PGPSize *		newobjslength
	)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_RevokeSig_MSG, connect_ref);
	pack_uint32(&pkt, sigid);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpRevokeKey_backRPC (
	PGPContextRef		context,
	PGPUInt32			key_id,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPBoolean			hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal,
	PGPUInt32 **		newobjs,
	PGPSize *			newobjslength
	)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_RevokeKey_MSG, connect_ref);
	pack_uint32(&pkt, key_id);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);


	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpDoChangePassphrase_backRPC (PGPContextRef context, PGPUInt32 keydb_id,
	PGPUInt32 key_id, PGPUInt32 masterkey_id, 
	const char *oldphrase, PGPSize oldphraseLength,
	const char *newphrase, PGPSize newphraseLength,
	PGPBoolean	newPassphraseIsKey,
	PGPUInt32	cacheTimeOut,
	PGPBoolean	cacheGlobal)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_DoChangePassphrase_MSG, connect_ref);
	pack_uint32(&pkt, keydb_id);
	pack_uint32(&pkt, key_id);
	pack_uint32(&pkt, masterkey_id);
	pack_opaque(&pkt, (void *)oldphrase, oldphraseLength);
	pack_opaque(&pkt, (void *)newphrase, newphraseLength);
	pack_bool(&pkt, newPassphraseIsKey);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpPassphraseIsValid_backRPC(
	PGPContextRef	context,
	PGPUInt32		key_id,
	const char *	passphrase,
	PGPSize			passphraseLength,
	PGPBoolean		hashedPhrase,
	PGPUInt32		cacheTimeOut,
	PGPBoolean		cacheGlobal,
	PGPBoolean *	isValid)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_PassphraseIsValid_MSG, connect_ref);
	pack_uint32(&pkt, key_id);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*isValid = unpack_bool(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpSetKeyTrust_backRPC(
	PGPContextRef	context,
	PGPUInt32		key_id,
	PGPUInt32		trust)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_SetKeyTrust_MSG, connect_ref);
	pack_uint32(&pkt, key_id);
	pack_uint32(&pkt, trust);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpGetPasskeyBuffer_backRPC (
	PGPContextRef		context,
	PGPUInt32			key_id,
	char const *		passphrase,
	PGPSize				passphraseLength,
	PGPByte				**passkeyBuffer,
	PGPSize				*passkeyBufferLength
	)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_GetPasskeyBuffer_MSG, connect_ref);
	pack_uint32(&pkt, key_id);
	pack_opaque(&pkt, (void *)passphrase, passphraseLength);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_secure(&rplypkt, (void **)passkeyBuffer, passkeyBufferLength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpAddKeyOptions_backRPC (PGPContextRef context, PGPUInt32 key_id,
	char * passphrase, PGPSize passphraseLength, PGPBoolean hashedPhrase,
	PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPUInt32 *raklist, PGPSize raklistsize, PGPUInt32 rakclass,
	PGPUInt32 **newobjs, PGPSize *newobjslength)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_AddKeyOptions_MSG, connect_ref);
	pack_uint32(&pkt, key_id);
	pack_opaque(&pkt, passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);
	pack_opaque(&pkt, raklist, raklistsize);
	pack_uint32(&pkt, rakclass);

	if (raklist) PGPFreeData(raklist);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpUpdateKeyOptions_backRPC (
	PGPContextRef	   context,
	PGPUInt32		   key_id,
	char *			   passphrase,
	PGPSize			   passphraseLength,
	PGPBoolean		   hashedPhrase,
	PGPUInt32			cacheTimeOut,
	PGPBoolean			cacheGlobal,
	PGPCipherAlgorithm	*prefalg,
	PGPSize				prefalgLength,
	PGPByte				*prefkeyserv,
	PGPSize				prefkeyservLength,
	PGPUInt32			keyflags,
	PGPBoolean			fkeyflags,
	PGPUInt32			keyservprefs,
	PGPBoolean			fkeyservprefs,
	PGPUInt32 **		newobjs,
	PGPSize *			newobjslength
	)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init_secure(&pkt, kPGP_UpdateKeyOptions_MSG, connect_ref);
	pack_uint32(&pkt, key_id);
	pack_opaque(&pkt, passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);
	pack_opaque(&pkt, prefalg, prefalgLength);
	pack_opaque(&pkt, prefkeyserv, prefkeyservLength);
	pack_uint32(&pkt, keyflags);
	pack_bool(&pkt, fkeyflags);
	pack_uint32(&pkt, keyservprefs);
	pack_bool(&pkt, fkeyservprefs);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpKeyDBAddObject_backRPC( PGPContextRef context, PGPUInt32 kdb_id,
	PGPUInt32 obj_id, PGPUInt32 **newobjs, PGPSize *newobjslength,
	PGPUInt32 *pnewobj )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_KeyDBAddObject_MSG, connect_ref);
	pack_uint32(&pkt, kdb_id);
	pack_uint32(&pkt, obj_id);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjslength);
		*pnewobj = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpKeyDBRemoveObject_backRPC( PGPContextRef context, PGPUInt32 kdb_id,
	PGPUInt32 obj_id )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_KeyDBRemoveObject_MSG, connect_ref);
	pack_uint32(&pkt, kdb_id);
	pack_uint32(&pkt, obj_id);
	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpCopyKeys_backRPC( PGPContextRef context, PGPUInt32 srcid,
	PGPUInt32 dstid, PGPUInt32 *keylist, PGPSize keylistsize,
	PGPBoolean neednewkeylist, PGPUInt32 **newkeylist, PGPSize *newkeylistsize)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_CopyKeys_MSG, connect_ref);
	pack_uint32(&pkt, srcid);
	pack_uint32(&pkt, dstid);
	pack_opaque(&pkt, keylist, keylistsize);
	pack_bool(&pkt, neednewkeylist);

	if (keylist) PGPFreeData(keylist);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		if( neednewkeylist )
			unpack_opaque_alloc(&rplypkt, (void **)newkeylist, newkeylistsize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}


	PGPError
pgpKeyDBArray_backRPC(
	PGPContextRef cdkContext,
	PGPUInt32 kdbid,
	PGPUInt32 * numKeys,
	PGPUInt32 ** keyArray,
	PGPSize * keyArraySize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_KeyDBArray_MSG, connect_ref);
	pack_uint32(&pkt, kdbid);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*numKeys = unpack_uint32(&rplypkt);
		unpack_opaque_alloc(&rplypkt, (void **)keyArray, keyArraySize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpUpdateKeyDB_backRPC(
	PGPContextRef cdkContext,
	PGPUInt32 kdbid,
	PGPUInt32 * numNewKeys,
	PGPUInt32 ** newKeyArray,
	PGPSize * newKeyArraySize,
	PGPUInt32 **changedkeylist,
	PGPSize *changedkeylistsize )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_UpdateKeyDB_MSG, connect_ref);
	pack_uint32(&pkt, kdbid);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*numNewKeys = unpack_uint32(&rplypkt);
		unpack_opaque_alloc(&rplypkt, (void **)newKeyArray, newKeyArraySize);
		unpack_opaque_alloc(&rplypkt, (void **)changedkeylist, changedkeylistsize);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpNewKeyDB_backRPC( PGPContextRef context, PGPUInt32 *keydbid )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_NewKeyDB_MSG, connect_ref);
	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		*keydbid = unpack_uint32(&rplypkt);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpCertifyPrimaryUserID_backRPC (PGPContextRef ctxt,
		PGPUInt32 useridid, char *passphrase,
		PGPSize passphraseLength, PGPBoolean hashedPhrase,
		PGPUInt32	cacheTimeOut,
		PGPBoolean	cacheGlobal,
		PGPUInt32 **newobjs, PGPSize *newobjsLength)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)ctxt;
	pack_init_secure(&pkt, kPGP_CertifyPrimaryUserID_MSG, connect_ref);
	pack_uint32(&pkt, useridid);
	pack_opaque(&pkt, passphrase, passphraseLength);
	pack_bool(&pkt, hashedPhrase);
	pack_uint32(&pkt, cacheTimeOut);
	pack_bool(&pkt, cacheGlobal);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)newobjs, newobjsLength);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpGlobalRandomPoolAddState_backRPC( PGPBoolean addKeyState,
		PGPInt32 keyEvent, PGPBoolean addMouseState, PGPBoolean addSystemState )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_GlobalRandomPoolAddState_MSG, connect_ref);
	pack_bool(&pkt, addKeyState);
	pack_int32(&pkt, keyEvent);
	pack_bool(&pkt, addMouseState);
	pack_bool(&pkt, addSystemState);

	status = pgpRPCSendPacket(&pkt, &rplypkt, FALSE);  /* okayToBlock == FALSE */
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

PGPError
pgpGlobalRandomPoolGetInfo_backRPC( PGPUInt32 *entropy, PGPUInt32 *size,
		PGPUInt32 *minEntropy, PGPBoolean *hasMinEntropy,
		PGPBoolean *hasIntelRNG )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_GlobalRandomPoolGetInfo_MSG, connect_ref);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPUInt32 ui; PGPBoolean b;
		ui = unpack_uint32(&rplypkt);
		if (entropy) *entropy = ui;

		ui = unpack_uint32(&rplypkt);
		if (size) *size = ui;

		ui = unpack_uint32(&rplypkt);
		if (minEntropy) *minEntropy = ui;

		b = unpack_bool(&rplypkt);
		if (hasMinEntropy) *hasMinEntropy = b;

		b = unpack_bool(&rplypkt);
		if (hasIntelRNG) *hasIntelRNG = b;

		PGPFreeData(rplypkt.base);
	}
	return status;
}

PGPError
pgpRandomGetBytesEntropy_backRPC( PGPContextRef context,
		PGPUInt32 requestBytes, PGPUInt32 entropyBits,
		PGPByte **randBuf, PGPSize *randBufLen )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_RandomGetBytesEntropy_MSG, connect_ref);
	pack_uint32(&pkt, requestBytes);
	pack_uint32(&pkt, entropyBits);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		unpack_opaque_alloc(&rplypkt, (void **)randBuf, randBufLen);
		PGPFreeData(rplypkt.base);
	}
	return status;
}

PGPError
pgpRandomAddBytes_backRPC( PGPContextRef context,
		PGPByte const *buf, PGPSize buflen )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_RandomAddBytes_MSG, connect_ref);
	pack_opaque(&pkt, buf, buflen);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

PGPError
pgpRandomStir_backRPC( PGPContextRef context )
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	(void)context;
	pack_init(&pkt, kPGP_RandomStir_MSG, connect_ref);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}

	PGPError
pgpSetRandSeedFile_backRPC(
	PFLFileSpecRef randFile)
{
	PGPInt32 status;
	PGPPackMsg pkt, rplypkt;

	pack_init(&pkt, kPGP_SetRandSeedFile_MSG, connect_ref);
	pack_filespec(&pkt, randFile);

	status = pgpRPCSendPacket(&pkt, &rplypkt, TRUE);
	if (IsntPGPError(status)) {
		PGPFreeData(rplypkt.base);
	}
	return status;
}


/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
